import { create } from "zustand";
import { loadPuzzles } from "../utils/puzzlesData";

const STORAGE_KEY = "puzzleport_run_history_v1";

export const usePuzzleStore = create((set, get) => ({
  levels: [],
  levelIndex: 0,
  streak: 0,
  correctCount: 0,
  wrongCount: 0,
  totalTime: 0,
  initialized: false,

  async initPuzzles() {
    const levels = await loadPuzzles();
    set({
      levels,
      levelIndex: 0,
      streak: 0,
      correctCount: 0,
      wrongCount: 0,
      totalTime: 0,
      initialized: true,
    });
  },

  markCorrect(timeTaken = 0) {
    set((state) => ({
      correctCount: state.correctCount + 1,
      streak: state.streak + 1,
      totalTime: state.totalTime + timeTaken,
    }));
  },

  markWrong(timeTaken = 0) {
    set((state) => ({
      wrongCount: state.wrongCount + 1,
      streak: 0,
      totalTime: state.totalTime + timeTaken,
    }));
  },

  nextQuestion() {
    set((state) => ({
      levelIndex: Math.min(state.levelIndex + 1, (state.levels || []).length),
    }));
  },

  saveRun(name = "Player") {
    const state = get();
    try {
      const raw = JSON.parse(localStorage.getItem(STORAGE_KEY) || "[]");
      raw.push({
        name,
        correct: state.correctCount,
        wrong: state.wrongCount,
        time: state.totalTime,
        date: Date.now(),
      });
      raw.sort((a, b) => b.correct - a.correct || a.time - b.time);
      localStorage.setItem(STORAGE_KEY, JSON.stringify(raw.slice(0, 50)));
    } catch (e) {
      console.error("Failed to save run", e);
    }
  },

  resetAll() {
    set({
      levelIndex: 0,
      streak: 0,
      correctCount: 0,
      wrongCount: 0,
      totalTime: 0,
    });
  },
}));
